/**
 * AugerPlus - Project Hide/Unhide Functionality
 * Manages hiding and unhiding projects in a table interface
 */

// Configuration constants
const CONFIG = {
	COOKIE_NAME: "augerHideHiddenProjects",
	COOKIE_MAX_AGE: 34560000,
	POLL_INTERVAL: 200,
	CSS_CLASSES: {
		HIDE_COLUMN: "augerHide-newColumn",
		HIDE_CELL: "augerHide-newCell",
		SHOW_HIDDEN_BTN: "augerHide-showHiddenBtn",
		MODAL: "augerHide-modal",
	},
	SELECTORS: {
		TABLE: '[role="table"]',
		TABLE_HEADER: "thead tr",
		TABLE_BODY: "tbody tr",
		TABLE_CELL: '[role="cell"]',
		CONTENT_CENTER: ".content-select-center",
	},
};

/**
 * Storage utility for managing hidden projects in cookies
 */
class ProjectStorage {
	static getHiddenProjects() {
		const match = document.cookie.match(new RegExp(`${CONFIG.COOKIE_NAME}=([^;]*)`));
		try {
			return match ? JSON.parse(decodeURIComponent(match[1])) : [];
		} catch (error) {
			console.warn("Failed to parse hidden projects from cookie:", error);
			return [];
		}
	}

	static setHiddenProjects(projectIds) {
		if (!Array.isArray(projectIds)) {
			console.error("Project IDs must be an array");
			return;
		}

		const cookieValue = encodeURIComponent(JSON.stringify(projectIds));
		document.cookie = `${CONFIG.COOKIE_NAME}=${cookieValue}; path=/; max-age=${CONFIG.COOKIE_MAX_AGE}`;
	}

	static addHiddenProject(projectId) {
		const hiddenProjects = this.getHiddenProjects();
		if (!hiddenProjects.includes(projectId)) {
			hiddenProjects.push(projectId);
			this.setHiddenProjects(hiddenProjects);
		}
	}

	static removeHiddenProject(projectId) {
		const hiddenProjects = this.getHiddenProjects();
		const updatedProjects = hiddenProjects.filter((id) => id !== projectId);
		this.setHiddenProjects(updatedProjects);
	}
}

/**
 * Utility functions for DOM manipulation
 */
class DOMUtils {
	static waitForElement(selector, callback, interval = CONFIG.POLL_INTERVAL) {
		const pollInterval = setInterval(() => {
			const element = document.querySelector(selector);
			if (element) {
				clearInterval(pollInterval);
				callback(element);
			}
		}, interval);
		return pollInterval;
	}

	static waitForTableReady(callback) {
		const pollInterval = setInterval(() => {
			const table = document.querySelector(CONFIG.SELECTORS.TABLE);
			const hasHeader = table?.querySelector(CONFIG.SELECTORS.TABLE_HEADER);
			const hasBody = table?.querySelector(CONFIG.SELECTORS.TABLE_BODY);

			if (table && hasHeader && hasBody) {
				clearInterval(pollInterval);
				callback(table);
			}
		}, CONFIG.POLL_INTERVAL);
		return pollInterval;
	}

	static createElement(tag, options = {}) {
		const element = document.createElement(tag);

		if (options.className) element.className = options.className;
		if (options.textContent) element.textContent = options.textContent;
		if (options.innerHTML) element.innerHTML = options.innerHTML;
		if (options.onclick) element.onclick = options.onclick;
		if (options.style) {
			if (typeof options.style === "string") {
				element.style.cssText = options.style;
			} else {
				Object.assign(element.style, options.style);
			}
		}

		return element;
	}
}

/**
 * Manages project table functionality
 */
class ProjectTableManager {
	constructor() {
		this.observers = new Set();
	}

	initialize() {
		DOMUtils.waitForTableReady((table) => {
			this.setupTable();
			this.observeTableChanges(table);
		});
	}

	setupTable() {
		const hiddenProjects = ProjectStorage.getHiddenProjects();
		const tables = document.querySelectorAll(CONFIG.SELECTORS.TABLE);

		tables.forEach((table) => {
			this.addHideColumnHeader(table);
			this.processTableRows(table, hiddenProjects);
		});
	}

	addHideColumnHeader(table) {
		const headerRow = table.querySelector(CONFIG.SELECTORS.TABLE_HEADER);
		if (!headerRow || headerRow.querySelector(`.${CONFIG.CSS_CLASSES.HIDE_COLUMN}`)) {
			return;
		}

		const headerCell = DOMUtils.createElement("th", {
			className: CONFIG.CSS_CLASSES.HIDE_COLUMN,
			textContent: "Hide Project",
			style: {
				position: "sticky",
				zIndex: "999",
				width: "150px",
				minWidth: "80px",
			},
		});

		headerRow.appendChild(headerCell);
	}

	processTableRows(table, hiddenProjects) {
		const rows = table.querySelectorAll(CONFIG.SELECTORS.TABLE_BODY);

		rows.forEach((row) => {
			const projectId = this.getProjectIdFromRow(row);

			if (projectId) {
				this.toggleRowVisibility(row, hiddenProjects.includes(projectId));
			}

			this.addHideButtonToRow(row, projectId);
		});
	}

	getProjectIdFromRow(row) {
		const firstCell = row.querySelector(CONFIG.SELECTORS.TABLE_CELL);
		return firstCell?.innerText.trim() || null;
	}

	toggleRowVisibility(row, shouldHide) {
		row.style.display = shouldHide ? "none" : "";
	}

	addHideButtonToRow(row, projectId) {
		if (row.querySelector(`.${CONFIG.CSS_CLASSES.HIDE_CELL}`)) {
			return;
		}

		const cell = DOMUtils.createElement("td", {
			className: CONFIG.CSS_CLASSES.HIDE_CELL,
			style: {
				position: "sticky",
				zIndex: "999",
			},
		});

		const hideButton = DOMUtils.createElement("a", {
			textContent: "Hide",
			className: "btn btn-sm btn-info btn-block",
			onclick: (event) => this.handleHideProject(event, row, projectId),
		});

		cell.appendChild(hideButton);
		row.appendChild(cell);
	}

	handleHideProject(event, row, projectId) {
		event.preventDefault();
		event.stopPropagation();

		if (!projectId) return;

		this.toggleRowVisibility(row, true);
		ProjectStorage.addHiddenProject(projectId);

		// Refresh the table to ensure consistency
		setTimeout(() => this.setupTable(), 0);
	}

	observeTableChanges(table) {
		// Clean up existing observers
		this.observers.forEach((observer) => observer.disconnect());
		this.observers.clear();

		const observer = new MutationObserver(() => {
			this.setupTable();
		});

		observer.observe(table, {
			childList: true,
			subtree: true,
		});

		this.observers.add(observer);
	}

	cleanup() {
		this.observers.forEach((observer) => observer.disconnect());
		this.observers.clear();
	}
}

/**
 * Manages the unhide button and modal functionality
 */
class UnhideManager {
	constructor(tableManager) {
		this.tableManager = tableManager;
	}

	initialize() {
		DOMUtils.waitForElement(CONFIG.SELECTORS.CONTENT_CENTER, (centerDiv) => {
			this.addUnhideButton(centerDiv);
		});
	}

	addUnhideButton(centerDiv) {
		if (centerDiv.querySelector(`.${CONFIG.CSS_CLASSES.SHOW_HIDDEN_BTN}`)) {
			return;
		}

		const buttonContainer = DOMUtils.createElement("div", {
			className: "content-select-create",
		});

		const unhideButton = DOMUtils.createElement("button", {
			className: `${CONFIG.CSS_CLASSES.SHOW_HIDDEN_BTN} btn btn-success btn-lg btn-block`,
			textContent: "UNHIDE",
			onclick: () => this.showModal(),
		});

		buttonContainer.appendChild(unhideButton);
		centerDiv.appendChild(buttonContainer);
	}

	showModal() {
		this.removeExistingModal();

		const modal = this.createModal();
		const modalContent = this.createModalContent();

		modal.appendChild(modalContent);
		document.body.appendChild(modal);
	}

	removeExistingModal() {
		const existingModal = document.querySelector(`.${CONFIG.CSS_CLASSES.MODAL}`);
		if (existingModal) {
			existingModal.remove();
		}
	}

	createModal() {
		return DOMUtils.createElement("div", {
			className: CONFIG.CSS_CLASSES.MODAL,
			style: {
				position: "fixed",
				left: "0",
				top: "0",
				width: "100vw",
				height: "100vh",
				background: "rgba(0,0,0,0.4)",
				zIndex: "9999",
				display: "flex",
				alignItems: "center",
				justifyContent: "center",
			},
		});
	}

	createModalContent() {
		const hiddenProjects = ProjectStorage.getHiddenProjects();

		const contentBox = DOMUtils.createElement("div", {
			style: {
				background: "#fff",
				padding: "24px",
				borderRadius: "8px",
				minWidth: "320px",
				maxWidth: "90vw",
				maxHeight: "80vh",
				overflowY: "auto",
			},
		});

		// Add title
		const title = DOMUtils.createElement("h3", {
			textContent: "Hidden Projects",
		});
		contentBox.appendChild(title);

		// Add content based on hidden projects
		if (hiddenProjects.length === 0) {
			this.addEmptyState(contentBox);
		} else {
			this.addProjectList(contentBox, hiddenProjects);
		}

		// Add close button
		this.addCloseButton(contentBox);

		return contentBox;
	}

	addEmptyState(container) {
		const emptyMessage = DOMUtils.createElement("div", {
			textContent: "No hidden projects.",
		});
		container.appendChild(emptyMessage);
	}

	addProjectList(container, hiddenProjects) {
		hiddenProjects.forEach((projectId) => {
			const projectRow = this.createProjectRow(projectId);
			container.appendChild(projectRow);
		});
	}

	createProjectRow(projectId) {
		const row = DOMUtils.createElement("div", {
			style: {
				display: "flex",
				alignItems: "center",
				marginBottom: "8px",
			},
		});

		const label = DOMUtils.createElement("span", {
			textContent: projectId,
			style: { flex: "1" },
		});

		const unhideButton = DOMUtils.createElement("button", {
			className: "btn btn-success btn-sm",
			textContent: "Unhide",
			onclick: () => this.handleUnhideProject(projectId),
		});

		row.appendChild(label);
		row.appendChild(unhideButton);

		return row;
	}

	handleUnhideProject(projectId) {
		ProjectStorage.removeHiddenProject(projectId);

		// Update table visibility immediately
		this.updateTableVisibility(projectId);

		// Refresh the table
		setTimeout(() => {
			this.tableManager.setupTable();
		}, 0);

		// Refresh the modal
		this.showModal();
	}

	updateTableVisibility(projectId) {
		const tableRows = document.querySelectorAll(`${CONFIG.SELECTORS.TABLE} ${CONFIG.SELECTORS.TABLE_BODY}`);

		tableRows.forEach((row) => {
			const rowProjectId = this.tableManager.getProjectIdFromRow(row);
			if (rowProjectId === projectId) {
				this.tableManager.toggleRowVisibility(row, false);
			}
		});
	}

	addCloseButton(container) {
		const closeButton = DOMUtils.createElement("button", {
			className: "btn btn-secondary btn-sm",
			textContent: "Close",
			style: { marginTop: "16px" },
			onclick: () => this.removeExistingModal(),
		});

		container.appendChild(closeButton);
	}
}

/**
 * Main application controller
 */
class AugerPlusApp {
	constructor() {
		this.tableManager = new ProjectTableManager();
		this.unhideManager = new UnhideManager(this.tableManager);
		this.isInitialized = false;
		this.lastUrl = location.href;
	}

	initialize() {
		if (this.isInitialized) return;

		this.tableManager.initialize();
		this.unhideManager.initialize();
		this.setupUrlChangeListener();

		this.isInitialized = true;
	}

	setupUrlChangeListener() {
		const urlObserver = new MutationObserver(() => {
			if (location.href !== this.lastUrl) {
				this.lastUrl = location.href;
				this.handleUrlChange();
			}
		});

		urlObserver.observe(document, {
			childList: true,
			subtree: true,
		});
	}

	handleUrlChange() {
		// Reinitialize components when URL changes
		this.tableManager.initialize();
		this.unhideManager.initialize();
	}

	cleanup() {
		this.tableManager.cleanup();
		this.isInitialized = false;
	}
}

// Initialize the application
const augerPlusApp = new AugerPlusApp();

// Start the application when the page loads
window.addEventListener("load", () => {
	augerPlusApp.initialize();
});

// Handle page unload
window.addEventListener("beforeunload", () => {
	augerPlusApp.cleanup();
});
